#ifndef CRYS_ECPKI_ECDSA_SEP_H
#define CRYS_ECPKI_ECDSA_SEP_H
/*
 *  Object name     :  CRYS__ECPKI_ECDSA_SEP.h
   *  State           :  %state%
   *  Creation date   :  19 Apr 2007
   *  Last modified   :  %modify_time%
   */
/** @file
 * \brief Defines the APIs that support the ECDSA functions with multi LLI DMA interface. 
 *
 * \version CRYS__ECPKI_ECDSA_SEP.h#1:hinc:1
 * \author R.Levin
 * \remarks Copyright (C) 2007 by Discretix Technologies Ltd. All rights reserved.
 */

#include "DX_VOS_BaseTypes.h"
#include "CRYS_error.h"
#include "CRYS_Defs.h"
#include "CRYS_ECPKI_Types.h"

#ifdef __cplusplus
extern "C"
{
#endif

 


/**************************************************************************
 *	              CRYS_ECDSA_SEP_Sign_Update function
 **************************************************************************/
/**
   @brief  Performs a hash  operation on data allocated by the user before signing it.
           Multi LLI DMA interface for data input is used. 
  
           In case the user divides signing data by block, the user must call the Update   
           continuously for some period, until processing of the entire data block is complete.
 
           NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
                 is not recommended!

   @param[in,out] SignUserContext_ptr - A pointer to the user buffer for signing the database.
   @param[in] DataInLliTab_ptr   - The pointer to the buffer of LLI table, containing pointers 
                                   and sizes of chuncks of input data block. 
                                   The table need to be aligned and placed in SEP SRAM.
   @param[in] InLliTabEntries    - The count of entries in the input LLI table.
   @param[in] DataInSize         - The size of the message data block, in bytes.
   
   @return <b>CRYSError_t</b>: <br> 
			 CRYS_OK<br>
			 CRYS_ECDSA_SIGN_INVALID_USER_CONTEXT_PTR_ERROR
			 CRYS_ECDSA_SIGN_USER_CONTEXT_VALIDATION_TAG_ERROR
			 CRYS_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR 
			 CRYS_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_SIZE_ERROR
			 CRYS_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR
		 										 
 */
CIMPORT_C CRYSError_t CRYS_ECDSA_SEP_SignUpdate(CRYS_ECDSA_SignUserContext_t  *SignUserContext_ptr,  /*in/out*/
												DxUint8_t                     *intDataHead_ptr,     
												DxUint32_t                    intDataHeadSize, 
												DxUint8_t                     *intDataTail_ptr,     
												DxUint32_t                    intDataTailSize, 
                                                DxUint32_t                    *DataInLliTab_ptr,     /*in*/
                                                DxUint32_t                    InLliTabEntries,      /*in*/ 
                                                DxUint32_t                    DataInSize            /*in*/ );



/**************************************************************************
 *	              CRYS_ECDSA_Sign - integrated function
 **************************************************************************/
/**
   @brief  Performs all of the ECDSA signing operations simultaneously.
           Multi LLI DMA interface for data input is used. 

           This function simply calls the Init, Update and Finish functions continuously.  
   	       This function's prototype is similar to the prototypes of the called functions 
   	       and includes all of their input and output arguments.
   
           NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
                 is not recommended!

   @param[in,out] SignUserContext_ptr - A pointer to the user buffer for signing database.
   @param[in]     SignerPrivKey_ptr   - A pointer to a user private key structure.    				
   @param[in]     HashMode            - The enumerator variable defines hash function to be used.                         
   @param[in]     DataInLliTab_ptr    - The pointer to the buffer of LLI table, containing 
                                        pointers and sizes ofchuncks of input data block. 
                                        The table need to be aligned and placed in SEP SRAM.
   @param[in]     InLliTabEntries     - The count of entries in the input LLI table.
   @param[in]     MessageSizeInBytes  - A size of block of message data in bytes. 
   @param[in]     SignatureOut_ptr    - A pointer to a buffer for output of signature.                         
   @param[in,out] SignatureOutSize_ptr- Size of user passed buffer for signature (in)
                                        and size of actual signature (out). The size of buffer 
									    must be not less than 2*OrderSizeInBytes.
   @return <b>CRYSError_t</b>: <br> 
			 CRYS_OK<br>
			 CRYS_ECDSA_SIGN_INVALID_USER_CONTEXT_PTR_ERROR<br>
			 CRYS_ECDSA_SIGN_USER_CONTEXT_VALIDATION_TAG_ERROR<br>
 		     CRYS_ECDSA_SIGN_INVALID_USER_PRIV_KEY_PTR_ERROR<br>
			 CRYS_ECDSA_SIGN_USER_PRIV_KEY_VALIDATION_TAG_ERROR<br>
			 CRYS_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR <br>                         
			 CRYS_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR<br> 
			 CRYS_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_SIZE_ERROR<br>
			 CRYS_ECDSA_SIGN_INVALID_SIGNATURE_OUT_PTR_ERROR<br>
			 CRYS_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR<br>
             CRYS_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_ERROR<br>			 
			 CRYS_ECDSA_SIGN_INVALID_DOMAIN_ID_ERROR <br>			 
**/
CIMPORT_C CRYSError_t CRYS_ECDSA_SEP_Sign(
					       CRYS_ECDSA_SignUserContext_t  *SignUserContext_ptr,     /*in/out*/
					       CRYS_ECPKI_UserPrivKey_t      *SignerPrivKey_ptr,       /*in*/
					       CRYS_ECPKI_HASH_OpMode_t       HashMode,                /*in*/
						   DxUint8_t                      *intDataHead_ptr,     
						   DxUint32_t                     intDataHeadSize, 
						   DxUint8_t                      *intDataTail_ptr,     
						   DxUint32_t                     intDataTailSize, 
					       DxUint32_t                     *DataInLliTab_ptr,        /*in*/
					       DxUint32_t                     InLliTabEntries,         /*in*/ 
				     	   DxUint32_t                     MessageSizeInBytes,      /*in*/
					       DxUint8_t                     *SignatureOut_ptr,        /*out*/ 
					       DxUint32_t                    *SignatureOutSize_ptr     /*in*/); 




/**************************************************************************
 *	              CRYS_ECDSA_VerifyUpdate function
 **************************************************************************/
/**
   @brief  Performs a hash  operation on data allocated by the user before verifying it.
           Multi LLI DMA interface for data input is used. 
  
   	       In case user divides signing data by block, he must call the Update function 
   	       continuously a number of times until processing of the entire data block is complete.
   	   
           NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
                 is not recommended!

   @param[in,out] VerifyUserContext_ptr A pointer to the user buffer for signing database.
   @param[in]     DataInLliTab_ptr    - The pointer to the buffer of LLI table, containing 
                                        pointers and sizes ofchuncks of input data block. 
                                        The table need to be aligned and placed in SEP SRAM.
   @param[in]     InLliTabEntries     - The count of entries in the input LLI table.
   @param[in]     DataInSize            The size of the message data block, in bytes.
   
   @return <b>CRYSError_t</b>: <br> 
			 CRYS_OK<br>
                         CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br>
			 CRYS_ECDSA_VERIFY_USER_CONTEXT_VALIDATION_TAG_ERROR <br>
			 CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR <br>
			 CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_SIZE_ERROR <br>
			 CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>
 **/
CIMPORT_C CRYSError_t CRYS_ECDSA_SEP_VerifyUpdate(CRYS_ECDSA_VerifyUserContext_t *VerifyUserContext_ptr, /*in/out*/
							                      DxUint8_t                     *intDataHead_ptr,     
							                      DxUint32_t                    intDataHeadSize, 
							                      DxUint8_t                     *intDataTail_ptr,     
							                      DxUint32_t                    intDataTailSize, 
							                      DxUint32_t                     *DataInLliTab_ptr,/*in*/
							                      DxUint32_t                      InLliTabEntries,/*in*/ 
							                      DxUint32_t                      DataInSize /*in*/ );



/**************************************************************************
 *	              CRYS_ECDSA_Verify integrated function
 **************************************************************************/
/**
   @brief  Performs all of the ECDSA verifying operations simultaneously.
           Multi LLI DMA interface for data input is used. 

           This function simply calls the Init, Update and Finish functions continuously.  
   	       This function's prototype is similar to the prototypes of the called functions 
   	       and includes all of their input and output arguments.
   
           NOTE: Using of HASH functions with HASH size great, than EC modulus size, 
                 is not recommended!
   	   

   @param[in]  VerifyUserContext_ptr - A pointer to the user buffer for signing the database.
   @param[in]  UserPublKey_ptr       - A pointer to a user public key structure.
   @param[in]  HashMode              - The enumerator variable defines the hash function to be used.
   @param[in]  DataInLliTab_ptr      - The pointer to the buffer of LLI table, containing 
                                       pointers and sizes ofchuncks of input data block. 
                                       The table need to be aligned and placed in SEP SRAM.
   @param[in]  InLliTabEntries       - The count of entries in the input LLI table.
   @param[in]  MessageSizeInBytes    - Size of block of message data in bytes.       
   @param[in]  SignatureIn_ptr       - A pointer to a buffer for output of signature. 
   @param[in]  SignatureSizeBytes    - Size of signature, in bytes (must be 2*OrderSizeInBytes).
                                       
   @return <b>CRYSError_t</b>: <br> 
              CRYS_OK <br>
              CRYS_ECDSA_VERIFY_INVALID_USER_CONTEXT_PTR_ERROR <br> 
              CRYS_ECDSA_VERIFY_USER_CONTEXT_VALIDATION_TAG_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_DOMAIN_ID_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR <br>
              CRYS_ECDSA_VERIFY_SIGNER_PUBL_KEY_VALIDATION_TAG_ERROR <br>
              CRYS_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR <br>                
              CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR <br>
              CRYS_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_SIZE_ERROR <br>						                        
              CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR <br>						 
              CRYS_ECDSA_VERIFY_INVALID_SIGNATURE_SIZE_ERROR <br>
**/
CIMPORT_C CRYSError_t CRYS_ECDSA_SEP_Verify ( 
                                          CRYS_ECDSA_VerifyUserContext_t *VerifyUserContext_ptr, /*in/out*/
									   	  CRYS_ECPKI_UserPublKey_t       *UserPublKey_ptr,        /*in*/
									      CRYS_ECPKI_HASH_OpMode_t        HashMode,               /*in*/
										  DxUint8_t                      *SignatureIn_ptr,        /*in*/
										  DxUint32_t                      SignatureSizeBytes,     /*in*/
										  DxUint8_t                       *intDataHead_ptr,     
										  DxUint32_t                      intDataHeadSize, 
										  DxUint8_t                       *intDataTail_ptr,     
										  DxUint32_t                      intDataTailSize, 
								          DxUint32_t                     *DataInLliTab_ptr,       /*in*/
								          DxUint32_t                      InLliTabEntries,        /*in*/ 
										  DxUint32_t                      MessageSizeInBytes      /*in*/ );


/**********************************************************************************************************/


#ifdef __cplusplus
}
#endif			

#endif
